-- ============================================
-- Sub2me Borehole Tech - Database Installation
-- ============================================
-- Run this file once to set up your database
-- ============================================

-- Create database (uncomment if needed)
-- CREATE DATABASE borehole_tech;
-- USE borehole_tech;

-- ============================================
-- Table: agents
-- ============================================
CREATE TABLE IF NOT EXISTS agents (
    id INT PRIMARY KEY AUTO_INCREMENT,
    agent_name VARCHAR(255) NOT NULL,
    phone VARCHAR(20) NOT NULL,
    whatsapp VARCHAR(20) NULL,
    email VARCHAR(255) NULL,
    address TEXT NULL,
    total_jobs INT DEFAULT 0,
    total_revenue DECIMAL(15,2) DEFAULT 0.00,
    outstanding_balance DECIMAL(15,2) DEFAULT 0.00,
    rating TINYINT DEFAULT 5,
    is_active BOOLEAN DEFAULT 1,
    notes TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ============================================
-- Table: projects
-- ============================================
CREATE TABLE IF NOT EXISTS projects (
    id INT PRIMARY KEY AUTO_INCREMENT,
    project_number VARCHAR(50) UNIQUE NULL,
    agent_id INT NOT NULL,
    location VARCHAR(255) NOT NULL,
    
    -- Depth and pricing
    depth_required INT NOT NULL,
    depth_achieved INT DEFAULT 0,
    rate_per_meter DECIMAL(10,2) DEFAULT 12500.00,
    
    -- Payment structure
    total_amount DECIMAL(15,2) NOT NULL,
    initial_payment_percent INT DEFAULT 70,
    initial_payment_amount DECIMAL(15,2) NOT NULL,
    initial_payment_paid BOOLEAN DEFAULT 0,
    initial_payment_date DATE NULL,
    initial_payment_reference VARCHAR(100) NULL,
    
    balance_amount DECIMAL(15,2) NOT NULL,
    balance_paid BOOLEAN DEFAULT 0,
    balance_payment_date DATE NULL,
    balance_payment_reference VARCHAR(100) NULL,
    
    -- Adjustments/concessions
    discount_amount DECIMAL(10,2) DEFAULT 0.00,
    discount_reason VARCHAR(255) NULL,
    additional_charges DECIMAL(10,2) DEFAULT 0.00,
    additional_charges_reason VARCHAR(255) NULL,
    
    -- Project details
    status ENUM('pending_payment','approved','in_progress','completed','fully_paid','cancelled') DEFAULT 'pending_payment',
    assigned_team VARCHAR(100) NULL,
    start_date DATE NULL,
    completion_date DATE NULL,
    
    -- Additional info
    notes TEXT NULL,
    created_by VARCHAR(100) NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (agent_id) REFERENCES agents(id) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ============================================
-- Table: expenses
-- ============================================
CREATE TABLE IF NOT EXISTS expenses (
    id INT PRIMARY KEY AUTO_INCREMENT,
    project_id INT NULL,
    expense_type ENUM('diesel','maintenance','staff_salary','staff_feeding','site_expense','gm_commission','ceo_commission','other') NOT NULL,
    amount DECIMAL(15,2) NOT NULL,
    description TEXT NULL,
    expense_date DATE NOT NULL,
    recorded_by VARCHAR(100) NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ============================================
-- Table: payments
-- ============================================
CREATE TABLE IF NOT EXISTS payments (
    id INT PRIMARY KEY AUTO_INCREMENT,
    project_id INT NOT NULL,
    payment_type ENUM('initial','balance','partial','refund') NOT NULL,
    amount DECIMAL(15,2) NOT NULL,
    payment_method ENUM('cash','bank_transfer','pos','cheque','other') DEFAULT 'cash',
    payment_reference VARCHAR(100) NULL,
    payment_date DATE NOT NULL,
    received_by VARCHAR(100) NULL,
    notes TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE RESTRICT
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ============================================
-- Table: inventory
-- ============================================
CREATE TABLE IF NOT EXISTS inventory (
    id INT PRIMARY KEY AUTO_INCREMENT,
    item_name VARCHAR(255) NOT NULL,
    category ENUM('casing','pump','tank','accessories','diesel','other') NOT NULL,
    quantity DECIMAL(10,2) DEFAULT 0,
    unit VARCHAR(50) DEFAULT 'units',
    unit_price DECIMAL(10,2) DEFAULT 0.00,
    reorder_level DECIMAL(10,2) DEFAULT 0,
    supplier VARCHAR(255) NULL,
    last_restocked DATE NULL,
    notes TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ============================================
-- Table: activity_logs
-- ============================================
CREATE TABLE IF NOT EXISTS activity_logs (
    id INT PRIMARY KEY AUTO_INCREMENT,
    project_id INT NULL,
    action VARCHAR(100) NOT NULL,
    details TEXT NULL,
    user VARCHAR(100) NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ============================================
-- Table: users (for login system)
-- ============================================
CREATE TABLE IF NOT EXISTS users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    username VARCHAR(100) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    full_name VARCHAR(255) NOT NULL,
    email VARCHAR(255) NULL,
    role ENUM('admin','manager','accountant','viewer') DEFAULT 'viewer',
    is_active BOOLEAN DEFAULT 1,
    last_login TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ============================================
-- Table: settings
-- ============================================
CREATE TABLE IF NOT EXISTS settings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    setting_key VARCHAR(100) UNIQUE NOT NULL,
    setting_value TEXT NOT NULL,
    description VARCHAR(255) NULL,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ============================================
-- Insert Default Data
-- ============================================

-- Default admin user (username: admin, password: admin123)
-- IMPORTANT: Change this password after first login!
INSERT INTO users (username, password, full_name, role) 
VALUES ('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Administrator', 'admin');

-- Default settings
INSERT INTO settings (setting_key, setting_value, description) VALUES
('company_name', 'Sub2me Borehole Tech', 'Company name'),
('default_rate_per_meter', '12500', 'Default rate per meter in Naira'),
('default_initial_payment_percent', '70', 'Default initial payment percentage'),
('diesel_price_per_liter', '990', 'Current diesel price per liter'),
('diesel_consumption_per_meter', '2.14', 'Diesel consumption per meter (150L / 70m)'),
('maintenance_percent', '15', 'Maintenance percentage of revenue'),
('gm_commission_per_meter', '500', 'General Manager commission per meter'),
('ceo_commission_per_meter', '1500', 'CEO commission per meter'),
('staff_daily_feeding', '2500', 'Daily feeding per staff'),
('staff_count', '6', 'Number of site workers'),
('staff_monthly_salary_per_person', '30000', 'Monthly salary per worker'),
('monthly_site_expenses', '133333', 'Monthly site expenses (annual/12)'),
('target_boreholes_per_month', '20', 'Monthly target number of boreholes');

-- Sample agents
INSERT INTO agents (agent_name, phone, whatsapp, rating) VALUES
('Baba Sule', '08012345678', '08012345678', 5),
('Alhaji Musa', '08098765432', '08098765432', 4),
('Mama Caro', '08056781234', '08056781234', 5);

-- Sample inventory items
INSERT INTO inventory (item_name, category, quantity, unit, unit_price, reorder_level) VALUES
('6-inch PVC Casing Pipe', 'casing', 50, 'meters', 8500, 20),
('8-inch PVC Casing Pipe', 'casing', 30, 'meters', 12000, 15),
('Submersible Pump 1HP', 'pump', 5, 'units', 85000, 2),
('Water Tank 1000L', 'tank', 3, 'units', 45000, 1),
('Diesel', 'diesel', 500, 'liters', 990, 200);

-- ============================================
-- Create Indexes for Performance
-- ============================================
CREATE INDEX idx_projects_agent ON projects(agent_id);
CREATE INDEX idx_projects_status ON projects(status);
CREATE INDEX idx_projects_dates ON projects(start_date, completion_date);
CREATE INDEX idx_payments_project ON payments(project_id);
CREATE INDEX idx_expenses_project ON expenses(project_id);
CREATE INDEX idx_expenses_date ON expenses(expense_date);
CREATE INDEX idx_activity_logs_project ON activity_logs(project_id);

-- ============================================
-- Installation Complete!
-- ============================================
-- Next steps:
-- 1. Edit config.php with your database credentials
-- 2. Login with: username=admin, password=admin123
-- 3. Change the admin password immediately!
-- ============================================
