<?php
require_once 'config.php';
requireLogin();

$page_title = 'Agents Management';

// Handle agent addition
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_agent'])) {
    $agent_name = sanitize($_POST['agent_name']);
    $phone = sanitize($_POST['phone']);
    $whatsapp = sanitize($_POST['whatsapp'] ?? '');
    $email = sanitize($_POST['email'] ?? '');
    $address = sanitize($_POST['address'] ?? '');
    
    $stmt = $conn->prepare("INSERT INTO agents (agent_name, phone, whatsapp, email, address) VALUES (?, ?, ?, ?, ?)");
    $stmt->bind_param("sssss", $agent_name, $phone, $whatsapp, $email, $address);
    
    if ($stmt->execute()) {
        setSuccess("Agent added successfully!");
        header("Location: agents.php");
        exit;
    } else {
        setError("Error adding agent");
    }
}

// Get all agents
$agents = $conn->query("
    SELECT a.*, 
           COUNT(p.id) as total_projects,
           SUM(CASE WHEN p.status IN ('completed', 'fully_paid') THEN p.total_amount ELSE 0 END) as total_revenue,
           SUM(CASE WHEN p.balance_paid = 0 THEN p.balance_amount ELSE 0 END) as outstanding
    FROM agents a
    LEFT JOIN projects p ON a.id = p.agent_id
    GROUP BY a.id
    ORDER BY a.agent_name
");

include 'header.php';
?>

<div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
    <div>
        <h1>Agents Management</h1>
        <p class="text-muted">Manage drilling agents and their performance</p>
    </div>
    <button class="btn btn-success" onclick="document.getElementById('addAgentForm').style.display='block'">
        + Add New Agent
    </button>
</div>

<!-- Add Agent Form (Hidden) -->
<div id="addAgentForm" class="card" style="display: none; max-width: 600px; margin-bottom: 20px;">
    <h3>Add New Agent</h3>
    <form method="POST" action="">
        <div class="form-group">
            <label>Agent Name *</label>
            <input type="text" name="agent_name" required>
        </div>
        
        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
            <div class="form-group">
                <label>Phone *</label>
                <input type="text" name="phone" required>
            </div>
            <div class="form-group">
                <label>WhatsApp</label>
                <input type="text" name="whatsapp">
            </div>
        </div>
        
        <div class="form-group">
            <label>Email</label>
            <input type="email" name="email">
        </div>
        
        <div class="form-group">
            <label>Address</label>
            <textarea name="address" rows="2"></textarea>
        </div>
        
        <button type="submit" name="add_agent" class="btn btn-success">Save Agent</button>
        <button type="button" class="btn btn-secondary" onclick="document.getElementById('addAgentForm').style.display='none'">Cancel</button>
    </form>
</div>

<!-- Agents List -->
<div class="card">
    <div class="table-responsive">
        <table>
            <thead>
                <tr>
                    <th>Agent Name</th>
                    <th>Contact</th>
                    <th>Total Projects</th>
                    <th>Total Revenue</th>
                    <th>Outstanding</th>
                    <th>Rating</th>
                    <th>Status</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php while ($agent = $agents->fetch_assoc()): ?>
                <tr>
                    <td><strong><?php echo htmlspecialchars($agent['agent_name']); ?></strong></td>
                    <td>
                        <?php echo $agent['phone']; ?><br>
                        <?php if ($agent['whatsapp']): ?>
                            <small><a href="https://wa.me/<?php echo preg_replace('/[^0-9]/', '', $agent['whatsapp']); ?>" target="_blank">WhatsApp</a></small>
                        <?php endif; ?>
                    </td>
                    <td><?php echo $agent['total_projects']; ?></td>
                    <td><?php echo formatMoney($agent['total_revenue']); ?></td>
                    <td class="<?php echo $agent['outstanding'] > 0 ? 'text-warning' : 'text-success'; ?>">
                        <?php echo formatMoney($agent['outstanding']); ?>
                    </td>
                    <td>
                        <?php 
                        for ($i = 0; $i < 5; $i++) {
                            echo $i < $agent['rating'] ? '⭐' : '☆';
                        }
                        ?>
                    </td>
                    <td>
                        <?php if ($agent['is_active']): ?>
                            <span class="badge badge-completed">Active</span>
                        <?php else: ?>
                            <span class="badge badge-cancelled">Inactive</span>
                        <?php endif; ?>
                    </td>
                    <td>
                        <a href="projects.php?agent=<?php echo $agent['id']; ?>" class="btn btn-primary btn-sm">View Projects</a>
                    </td>
                </tr>
                <?php endwhile; ?>
            </tbody>
        </table>
    </div>
</div>

<?php include 'footer.php'; ?>
