<?php
require_once 'config.php';
requireLogin();

$page_title = 'Add New Project';

// Get agents list
$agents = $conn->query("SELECT * FROM agents WHERE is_active = 1 ORDER BY agent_name");

// Get default settings
$default_rate = getSetting('default_rate_per_meter', 12500);
$default_percent = getSetting('default_initial_payment_percent', 70);

include 'header.php';
?>

<h1>Add New Project</h1>
<p class="text-muted">Create a new borehole drilling project</p>

<div class="card" style="max-width: 900px;">
    <form id="projectForm" method="POST" action="save_project.php">
        
        <!-- Basic Info -->
        <div style="background: #f8f9fa; padding: 20px; margin: -20px -20px 20px -20px; border-radius: 8px 8px 0 0;">
            <h3 style="margin-bottom: 15px;">Project Details</h3>
            
            <div class="form-group">
                <label>Agent/Driller *</label>
                <select name="agent_id" required>
                    <option value="">Select Agent...</option>
                    <?php while ($agent = $agents->fetch_assoc()): ?>
                        <option value="<?php echo $agent['id']; ?>">
                            <?php echo htmlspecialchars($agent['agent_name']); ?> - <?php echo $agent['phone']; ?>
                        </option>
                    <?php endwhile; ?>
                </select>
                <small>Don't see the agent? <a href="agents.php">Add new agent</a></small>
            </div>
            
            <div class="form-group">
                <label>Location *</label>
                <input type="text" name="location" placeholder="e.g., Ibara, Abeokuta" required>
            </div>
            
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                <div class="form-group">
                    <label>Depth Required (meters) *</label>
                    <input type="number" name="depth_required" id="depth_required" 
                           value="70" min="1" required>
                </div>
                
                <div class="form-group">
                    <label>Rate per Meter (₦) *</label>
                    <input type="number" name="rate_per_meter" id="rate_per_meter" 
                           value="<?php echo $default_rate; ?>" min="1" step="0.01" required>
                </div>
            </div>
            
            <div style="background: #e3f2fd; padding: 15px; border-radius: 4px; border-left: 4px solid #2196F3;">
                <strong>Base Total:</strong> 
                <span id="base_total" style="font-size: 1.2rem; color: #1976D2; font-weight: bold;">₦0</span>
            </div>
        </div>
        
        <!-- Payment Structure -->
        <div style="background: #f8f9fa; padding: 20px; margin: 20px -20px; border-radius: 4px;">
            <h3 style="margin-bottom: 15px;">Payment Terms</h3>
            
            <div style="margin-bottom: 20px;">
                <label style="display: inline; margin-right: 20px;">
                    <input type="radio" name="payment_mode" value="percentage" checked 
                           onchange="togglePaymentMode()"> 
                    Percentage Split (Standard)
                </label>
                <label style="display: inline;">
                    <input type="radio" name="payment_mode" value="manual" 
                           onchange="togglePaymentMode()"> 
                    Custom Amount
                </label>
            </div>
            
            <!-- Percentage Mode -->
            <div id="percentage_mode">
                <div style="display: grid; grid-template-columns: 2fr 1fr; gap: 15px;">
                    <div class="form-group">
                        <label>Initial Payment (%) *</label>
                        <input type="number" name="initial_payment_percent" 
                               id="initial_payment_percent" 
                               value="<?php echo $default_percent; ?>" min="0" max="100">
                        <small>Common: 50%, 60%, 70%, 80%, or 100% for full payment</small>
                    </div>
                    
                    <div class="form-group">
                        <label>Balance (%)</label>
                        <input type="number" id="balance_percent" readonly 
                               style="background: #e9ecef; cursor: not-allowed;">
                    </div>
                </div>
                
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; background: #e8f5e9; padding: 15px; border-radius: 4px; margin-top: 10px;">
                    <div>
                        <strong style="color: #2e7d32;">Initial Payment Amount:</strong><br>
                        <span id="initial_amount_display" style="font-size: 1.3rem; color: #1b5e20; font-weight: bold;">₦0</span>
                    </div>
                    <div>
                        <strong style="color: #2e7d32;">Balance Amount:</strong><br>
                        <span id="balance_amount_display" style="font-size: 1.3rem; color: #1b5e20; font-weight: bold;">₦0</span>
                    </div>
                </div>
            </div>
            
            <!-- Manual Mode -->
            <div id="manual_mode" style="display: none;">
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                    <div class="form-group">
                        <label>Initial Payment (₦) *</label>
                        <input type="number" name="manual_initial_payment" 
                               id="manual_initial_payment" 
                               step="0.01" min="0">
                    </div>
                    
                    <div class="form-group">
                        <label>Balance (₦)</label>
                        <input type="number" id="manual_balance" readonly 
                               style="background: #e9ecef; cursor: not-allowed;">
                    </div>
                </div>
                
                <small style="color: #666;">
                    💡 Enter any custom amount for initial payment. Balance will auto-calculate.
                </small>
            </div>
            
            <input type="hidden" name="initial_payment_amount" id="initial_payment_amount">
            <input type="hidden" name="balance_amount" id="balance_amount">
            <input type="hidden" name="total_amount" id="total_amount">
        </div>
        
        <!-- Adjustments/Concessions -->
        <div style="background: #fff3e0; padding: 20px; margin: 20px -20px; border-radius: 4px;">
            <h3 style="margin-bottom: 15px;">Adjustments (Optional)</h3>
            
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                <div>
                    <div class="form-group">
                        <label>Discount Amount (₦)</label>
                        <input type="number" name="discount_amount" id="discount_amount" 
                               value="0" step="0.01" min="0">
                    </div>
                    <div class="form-group">
                        <label>Discount Reason</label>
                        <input type="text" name="discount_reason" 
                               placeholder="e.g., Loyal agent, bulk deal, referral bonus">
                    </div>
                </div>
                
                <div>
                    <div class="form-group">
                        <label>Additional Charges (₦)</label>
                        <input type="number" name="additional_charges" id="additional_charges" 
                               value="0" step="0.01" min="0">
                    </div>
                    <div class="form-group">
                        <label>Charge Reason</label>
                        <input type="text" name="additional_charges_reason" 
                               placeholder="e.g., Extra casing, rocky terrain, weekend work">
                    </div>
                </div>
            </div>
            
            <div id="adjusted_total_section" style="display: none; background: #ffecb3; padding: 15px; border-radius: 4px; margin-top: 10px; border-left: 4px solid #ffa000;">
                <strong style="color: #e65100;">Adjusted Total:</strong> 
                <span id="adjusted_total" style="font-size: 1.3rem; color: #e65100; font-weight: bold;">₦0</span>
            </div>
        </div>
        
        <!-- Other Details -->
        <div style="padding: 20px 0;">
            <h3 style="margin-bottom: 15px;">Additional Information</h3>
            
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                <div class="form-group">
                    <label>Assigned Team</label>
                    <select name="assigned_team">
                        <option value="">Not assigned yet</option>
                        <option value="Team A">Team A</option>
                        <option value="Team B">Team B</option>
                        <option value="Team C">Team C</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label>Expected Start Date</label>
                    <input type="date" name="start_date" min="<?php echo date('Y-m-d'); ?>">
                </div>
            </div>
            
            <div class="form-group">
                <label>Notes</label>
                <textarea name="notes" rows="3" 
                          placeholder="Any special instructions, client name, or additional details..."></textarea>
            </div>
        </div>
        
        <div style="border-top: 2px solid #e9ecef; padding-top: 20px; display: flex; gap: 10px;">
            <button type="submit" class="btn btn-success" style="flex: 1;">
                💾 Save Project
            </button>
            <a href="index.php" class="btn btn-secondary">Cancel</a>
        </div>
    </form>
</div>

<script>
function calculatePayments() {
    const depth = parseFloat(document.getElementById('depth_required').value) || 0;
    const rate = parseFloat(document.getElementById('rate_per_meter').value) || 0;
    const discount = parseFloat(document.getElementById('discount_amount').value) || 0;
    const additional = parseFloat(document.getElementById('additional_charges').value) || 0;
    
    // Calculate base total
    const baseTotal = depth * rate;
    document.getElementById('base_total').textContent = '₦' + baseTotal.toLocaleString('en-NG', {minimumFractionDigits: 2, maximumFractionDigits: 2});
    
    // Calculate adjusted total
    const adjustedTotal = baseTotal - discount + additional;
    document.getElementById('total_amount').value = adjustedTotal.toFixed(2);
    
    // Show/hide adjusted total section
    if (discount > 0 || additional > 0) {
        document.getElementById('adjusted_total_section').style.display = 'block';
        document.getElementById('adjusted_total').textContent = '₦' + adjustedTotal.toLocaleString('en-NG', {minimumFractionDigits: 2, maximumFractionDigits: 2});
    } else {
        document.getElementById('adjusted_total_section').style.display = 'none';
    }
    
    // Calculate payment split based on mode
    const paymentMode = document.querySelector('input[name="payment_mode"]:checked').value;
    
    if (paymentMode === 'percentage') {
        const initialPercent = parseFloat(document.getElementById('initial_payment_percent').value) || 0;
        const balancePercent = 100 - initialPercent;
        
        document.getElementById('balance_percent').value = balancePercent;
        
        const initialAmount = (adjustedTotal * initialPercent) / 100;
        const balanceAmount = adjustedTotal - initialAmount;
        
        document.getElementById('initial_amount_display').textContent = '₦' + initialAmount.toLocaleString('en-NG', {minimumFractionDigits: 2, maximumFractionDigits: 2});
        document.getElementById('balance_amount_display').textContent = '₦' + balanceAmount.toLocaleString('en-NG', {minimumFractionDigits: 2, maximumFractionDigits: 2});
        
        document.getElementById('initial_payment_amount').value = initialAmount.toFixed(2);
        document.getElementById('balance_amount').value = balanceAmount.toFixed(2);
    }
}

function calculateManualBalance() {
    const total = parseFloat(document.getElementById('total_amount').value) || 0;
    const initialPayment = parseFloat(document.getElementById('manual_initial_payment').value) || 0;
    const balance = total - initialPayment;
    
    document.getElementById('manual_balance').value = balance.toFixed(2);
    
    // Update hidden fields
    document.getElementById('initial_payment_amount').value = initialPayment.toFixed(2);
    document.getElementById('balance_amount').value = balance.toFixed(2);
    
    // Calculate percentage for storage
    const percent = total > 0 ? Math.round((initialPayment / total) * 100) : 0;
    document.getElementById('initial_payment_percent').value = percent;
}

function togglePaymentMode() {
    const paymentMode = document.querySelector('input[name="payment_mode"]:checked').value;
    
    if (paymentMode === 'percentage') {
        document.getElementById('percentage_mode').style.display = 'block';
        document.getElementById('manual_mode').style.display = 'none';
        calculatePayments();
    } else {
        document.getElementById('percentage_mode').style.display = 'none';
        document.getElementById('manual_mode').style.display = 'block';
        
        // Pre-fill with calculated amount
        const total = parseFloat(document.getElementById('total_amount').value) || 0;
        const initialPercent = parseFloat(document.getElementById('initial_payment_percent').value) || 70;
        document.getElementById('manual_initial_payment').value = ((total * initialPercent) / 100).toFixed(2);
        calculateManualBalance();
    }
}

// Initialize calculations on page load
document.addEventListener('DOMContentLoaded', function() {
    calculatePayments();
    
    // Add event listeners
    document.getElementById('depth_required').addEventListener('input', calculatePayments);
    document.getElementById('rate_per_meter').addEventListener('input', calculatePayments);
    document.getElementById('initial_payment_percent').addEventListener('input', calculatePayments);
    document.getElementById('discount_amount').addEventListener('input', calculatePayments);
    document.getElementById('additional_charges').addEventListener('input', calculatePayments);
    document.getElementById('manual_initial_payment').addEventListener('input', calculateManualBalance);
});
</script>

<?php include 'footer.php'; ?>
